﻿using Ozeki;
using Ozeki.Common;
using Ozeki.Network;
using Ozeki.VoIP;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Security.Cryptography.X509Certificates;
using System.Text;
using System.Threading.Tasks;

namespace WhatsAppPBX
{
    internal class MyPBX : PBXBase
    {
        public const string UserAgent = "WhatsAppPBX-example";
        public const string TlsCertificatePath = "combined.pfx";
        public const string TlsCertificatePass = "qwe123";
        public const string BusinessPhoneNumber = "+00000000000";
        public const string SipUserPassword = "";
        public const string ExtensionId = "1000";
        public const string ExtensionPassword = "1000";
        public const string Hostname = "ozeki.hu";
        public const string VoIPServiceConnectionID = "VoIPConnection";
        public const string VoIPServiceOutboundPrefix = "+";

        string _localAddress;

        public MyPBX(string localAddress, int minPort, int maxPort) : base(minPort, maxPort, UserAgent)
        {
            _localAddress = localAddress;
            InitLogger();
            Console.WriteLine("PBX starting...");
            Console.WriteLine("Local address: " + localAddress);
        }

        protected override void OnStart()
        {
            ConfigureDialplanProvider();
            InitVoIpProvider();
            ConfigureTls();

            SetListenPort(_localAddress, 5060, TransportType.Udp);
            Console.WriteLine("Listened port: 5060 (UDP)");
            SetListenPort(_localAddress, 5061, TransportType.Tls);
            Console.WriteLine("Listened port: 5061 (TLS)");

            Console.WriteLine("PBX started.");
            base.OnStart();
        }

        protected override AuthenticationResult OnAuthenticationRequest(ISIPExtension extension, RequestAuthenticationInfo requestAuthInfo)
        {
            Console.WriteLine("Authentication request received from: " + extension.ExtensionID);
            var businessAuthName = BusinessPhoneNumber.TrimStart('+');
            if (extension.AuthName == businessAuthName)
                return extension.CheckPassword(extension.AuthName, SipUserPassword, requestAuthInfo);
            if (extension.ExtensionID == ExtensionId)
                return extension.CheckPassword(extension.AuthName, ExtensionPassword, requestAuthInfo);
            Console.WriteLine("Authentication failed. Unknown extension: " + extension.ExtensionID);
            return new AuthenticationResult(false, "Unknown extension.");
        }

        protected override RegisterResult OnRegisterReceived(ISIPExtension extension, SIPAddress from, int expires)
        {
            Console.WriteLine("Register received from: " + extension.ExtensionID);
            return base.OnRegisterReceived(extension, from, expires);
        }

        protected override void OnUnregisterReceived(ISIPExtension extension)
        {
            Console.WriteLine("Unregister received from: " + extension.ExtensionID);
            base.OnUnregisterReceived(extension);
        }

        protected override void OnCallRequestReceived(ISessionCall call)
        {
            Console.WriteLine("Call request received. Caller: " + call.DialInfo.CallerID + ", Dialed: " + call.DialInfo.Dialed);
            base.OnCallRequestReceived(call);
        }

        void ConfigureDialplanProvider()
        {
            var extensionContainer = GetService<IExtensionContainer>();
            var callManager = GetService<ICallManager>();
            callManager.DialplanProvider = new MyDialplanProvider(extensionContainer);
        }

        void InitVoIpProvider()
        {
            // Create VoIP provider connection for business-initiated calls
            var registerName = BusinessPhoneNumber.TrimStart('+');
            var registerPassword = SipUserPassword;
            var account = new SIPAccount(false, BusinessPhoneNumber, BusinessPhoneNumber, registerName, registerPassword, "wa.meta.vc", 5061);

            var natConfig = new NatConfiguration();
            natConfig.AutoDetect = false;
            natConfig.TraversalMethodType = NatTraversalMethod.STUN;
            //natConfig.StaticPublicAddress = "";

            var outboundCallMapping = new OutboundCallHeaderMapping();
            outboundCallMapping.AddMapping(new SIPHeaderMapping(SIPMappingField.From_User, CallProperty.CustomValue, BusinessPhoneNumber));
            outboundCallMapping.AddMapping(new SIPHeaderMapping(SIPMappingField.From_Host, CallProperty.CustomValue, Hostname));

            var phoneLineConfig = new PhoneLineConfiguration(account);
            phoneLineConfig.NatConfig = natConfig;
            phoneLineConfig.OutboundCallMapping = outboundCallMapping;
            phoneLineConfig.SRTPMode = SRTPMode.Force;
            phoneLineConfig.TransportType = TransportType.Tls;

            var extensionFactory = GetService<IExtensionFactory>();
            var sipClient = extensionFactory.CreateSIPClientExtension(VoIPServiceConnectionID, phoneLineConfig);
            var voIpProvider = new VoIPServiceProviderExtension(VoIPServiceConnectionID, sipClient);
            voIpProvider.RegistrationStateChanged += VoIpProvider_RegistrationStateChanged;

            // Add to container to make callable
            var extensionContainer = GetService<IExtensionContainer>();
            extensionContainer.TryAddExtension(voIpProvider);
            Console.WriteLine($"VoIP service provider connection added. SIP account: {account.UserName}@{account.DomainServerHost}");

            // Register VoIP provider
            extensionFactory.RegisterSIPClientExtension(voIpProvider);
        }

        void ConfigureTls()
        {
            try
            {
                if (!File.Exists(TlsCertificatePath))
                    throw new FileNotFoundException("TLS certificate file not found at: " + TlsCertificatePath);
                var cert = new X509Certificate2(TlsCertificatePath, TlsCertificatePass);
                var certInfo = new CertificateInfo(null, cert);
                var configurator = GetService<IConfigurator>();
                configurator.Certificates = new List<CertificateInfo> { certInfo };
                Console.WriteLine("TLS configuration completed.");
            }
            catch (Exception ex)
            {
                Console.WriteLine("TLS configuration error: " + ex.Message);
            }
        }

        void VoIpProvider_RegistrationStateChanged(object? sender, RegistrationStateChangedArgs e)
        {
            Console.WriteLine("VoIP service provider connection state changed: " + e.State);
        }

        void InitLogger()
        {
            try
            {
                var path = Path.Combine(AppDomain.CurrentDomain.BaseDirectory, "log.txt");
                var config = new LogConfig(LogLevel.Error) { MediaLogLevel = LogLevel.Debug, CallLogLevel = LogLevel.Debug };
                Logger.Attach(new DefaultFileLogger(path), config);
                Logger.Open();
            }
            catch (Exception ex)
            {
                Console.Error.WriteLine($"Logger initialization failed: {ex.Message}");
            }
        }
    }
}
